#!/usr/bin/env python3
import os
import sys
import shutil
import zipfile
import subprocess
import datetime
import yaml
import re
import wx
import io
import zlib  # <-- Added to calculate CRC32

# Ensure proper encoding (UTF-8) for all outputs and file interactions
if sys.version_info.major == 3:
    if sys.stdout is not None:
        sys.stdout = io.TextIOWrapper(sys.stdout.buffer, encoding='utf-8')
    if sys.stderr is not None:
        sys.stderr = io.TextIOWrapper(sys.stderr.buffer, encoding='utf-8')

# ----- Custom Quoted String for YAML values -----
class QuotedStr(str):
    pass

def quoted_str_representer(dumper, data):
    return dumper.represent_scalar('tag:yaml.org,2002:str', data, style='"')

yaml.add_representer(QuotedStr, quoted_str_representer)

def quote_values(obj):
    if isinstance(obj, dict):
        return {k: quote_values(v) for k, v in obj.items()}
    elif isinstance(obj, list):
        return [quote_values(i) for i in obj]
    elif isinstance(obj, str):
        return QuotedStr(obj)
    else:
        return obj

# Check if the script is running as a standalone executable
if getattr(sys, 'frozen', False):
    # If running as an executable, use the directory where the .exe is located
    current_directory = os.path.dirname(sys.executable)
else:
    # If running as a script, use the directory where the script is located
    current_directory = os.path.dirname(os.path.abspath(__file__))

# Define paths to the resources relative to the current directory
master_list_yaml = os.path.join(current_directory, 'master_list.yaml')
blanks_zip = os.path.join(current_directory, 'BLANKS.zip')
mymcplusplus_path = os.path.join(current_directory, 'mymcplusplus.exe')
ps2vmc_tool_path = os.path.join(current_directory, 'ps2vmc-tool.exe')

# Define TEMP_DIR (Temporary directory for working files)
TEMP_DIR = os.path.join(current_directory, 'temp')

# Print paths for debugging (optional)
print(f"master_list_yaml path: {master_list_yaml}")
print(f"BLANKS.zip path: {blanks_zip}")
print(f"mymcplusplus.exe path: {mymcplusplus_path}")
print(f"ps2vmc-tool.exe path: {ps2vmc_tool_path}")

if not os.path.exists(TEMP_DIR):
    os.makedirs(TEMP_DIR)

# ----- HELPER FUNCTION TO SANITIZE NAMES -----
def sanitize_name(name):
    name = name.replace("//", "").strip()
    name = re.sub(r'[^a-zAZ0-9 _-]', '', name)  # Only allow alphanumeric, spaces, underscores, and dashes
    return name

# ----- WX HELPER FUNCTIONS -----
def prompt_directory(message="Select a directory"):
    app = wx.GetApp() or wx.App(False)
    with wx.DirDialog(None, message, style=wx.DD_DEFAULT_STYLE) as dlg:
        if dlg.ShowModal() == wx.ID_OK:
            return dlg.GetPath()
    return None

def prompt_savefile():
    app = wx.GetApp() or wx.App(False)
    wildcard = "PS2 Save Files (*.psu;*.max;*.sps;*.xps;*.cbs;*.psv)|*.psu;*.max;*.sps;*.xps;*.cbs;*.psv"
    with wx.FileDialog(None, "Select a save file", wildcard=wildcard,
                       style=wx.FD_OPEN | wx.FD_FILE_MUST_EXIST) as dlg:
        if dlg.ShowModal() == wx.ID_OK:
            return dlg.GetPath()
    return None

def prompt_text(message, caption="Input", default_value=""):
    return wx.GetTextFromUser(message, caption, default_value)

def choose_executable():
    if os.path.exists(mymcplusplus_path):
        return mymcplusplus_path
    else:
        return "mymcplusplus"

# ----- NEW FUNCTION TO CALCULATE CRC32 -----
def calculate_crc32(file_path):
    """Calculate CRC32 for a given file."""
    hash_crc32 = 0
    with open(file_path, 'rb') as f:
        while chunk := f.read(8192):  # Read the file in chunks
            hash_crc32 = zlib.crc32(chunk, hash_crc32)
    return hash_crc32 & 0xFFFFFFFF  # Ensure the result is unsigned

# ----- RECURSIVE FUNCTION TO SEARCH FOR .bin, .ps2, .mc2 FILES -----
def find_files_recursive(base_folder, extensions=("bin", "ps2", "mc2")):
    """Recursively find files with the given extensions in the specified directory."""
    files_found = []
    for root, dirs, files in os.walk(base_folder):
        for file in files:
            if file.lower().endswith(tuple(f".{ext}" for ext in extensions)):
                files_found.append(os.path.join(root, file))
    return files_found

# ----- HELPER FUNCTIONS (unchanged) -----
def find_next_available_folder(base_folder):
    existing_folders = [f for f in os.listdir(base_folder) if os.path.isdir(os.path.join(base_folder, f)) and f.startswith("01")]
    numbers = []
    for folder in existing_folders:
        try:
            num = int(folder.split("_")[-1])
            numbers.append(num)
        except ValueError:
            continue

    return max(numbers, default=0) + 1

def extract_blank(file_name):
    with zipfile.ZipFile(blanks_zip, 'r') as z:
        z.extract(file_name, TEMP_DIR)
    return os.path.join(TEMP_DIR, file_name)

def run_command(command_list):
    try:
        command_list = [str(arg).encode('utf-8').decode('utf-8') if isinstance(arg, str) else arg for arg in command_list]
        result = subprocess.run(
            command_list,
            stdout=subprocess.PIPE,
            stderr=subprocess.PIPE,
            text=True,
            encoding="utf-8",
            errors="replace",
            check=True
        )
        return result.stdout
    except subprocess.CalledProcessError as e:
        print("Error running command:", " ".join(command_list))
        print(e.stderr)
        sys.exit(1)

def clean_serial(full_log_line):
    match = re.search(r"to\s+(.+)", full_log_line)
    if not match:
        print("Could not find 'to' in the log line")
        sys.exit(1)
    full_name = match.group(1).strip()
    cleaned = full_name[2:12]
    return cleaned, full_name

def safe_rename(src, dst):
    if not os.path.exists(src):
        print(f"Error: Source file {src} does not exist.")
        return
    
    print(f"Renaming file from {src} to {dst}")
    dst_dir = os.path.dirname(dst)
    if not os.path.exists(dst_dir):
        print(f"Creating destination directory: {dst_dir}")
        os.makedirs(dst_dir)
    
    if os.path.exists(dst):
        print(f"Destination file exists. Removing it: {dst}")
        os.remove(dst)
    
    try:
        shutil.move(src, dst)
        print(f"Successfully renamed {src} to {dst}")
    except Exception as e:
        print(f"Error moving file {src} to {dst}: {e}")

# ----- New function to handle moving folders to "complete" -----
def move_to_complete_folder(src_folder, base_complete_folder):
    if not os.path.exists(base_complete_folder):
        os.makedirs(base_complete_folder)

    base_folder_name = os.path.basename(src_folder)
    dst_folder = os.path.join(base_complete_folder, base_folder_name)

    if os.path.exists(dst_folder):
        print(f"Destination folder {dst_folder} already exists, renaming it to avoid conflict.")
        folder_name_parts = base_folder_name.split('_')
        if len(folder_name_parts) > 1 and folder_name_parts[-1].isdigit():
            counter = int(folder_name_parts[-1]) + 1
            new_folder_name = f"{'_'.join(folder_name_parts[:-1])}_{counter}"
        else:
            new_folder_name = f"{base_folder_name}_1"

        dst_folder = os.path.join(base_complete_folder, new_folder_name)
        print(f"Renamed destination folder to {dst_folder}")

    try:
        shutil.move(src_folder, dst_folder)
        print(f"Successfully moved {src_folder} to {dst_folder}")
    except Exception as e:
        print(f"Error moving folder {src_folder} to {dst_folder}: {e}")

# ----- MAIN PROCESS -----
def main():
    app = wx.App(False)

    MYMCPLUSPLUS = choose_executable()
    if not MYMCPLUSPLUS:
        print("No executable selected. Exiting...")
        sys.exit(1)

    save_file_path = prompt_savefile()
    if not save_file_path:
        print("No save file selected. Exiting...")
        sys.exit(1)

    # Default values for username, description, and site
    username = prompt_text("Enter your username:", "Username", "Anonymous")
    description = prompt_text("Enter game description:", "Game Description", "none")
    save_link = prompt_text("Enter save link (optional):", "Save Link", "none")
    date_str = datetime.datetime.now().strftime("%m/%d/%Y")

    with open(master_list_yaml, 'r', encoding='utf-8', errors='replace') as f:
        master_list = yaml.safe_load(f)

    blank_mc2 = extract_blank("BLANK.mc2")
    save_ext = os.path.splitext(save_file_path)[1]
    temp_save_path = os.path.join(TEMP_DIR, "0" + save_ext)
    shutil.copy(save_file_path, temp_save_path)

    cmd_import_mc2 = [MYMCPLUSPLUS, blank_mc2, "import", temp_save_path]
    output_import = run_command(cmd_import_mc2)
    print("MC2 Import Output:", output_import)

    cleaned_serial, save_directory_full = clean_serial(output_import)
    print("Cleaned serial:", cleaned_serial)
    print("Full SAVE_DIRECTORY:", save_directory_full)

    game_meta = master_list.get(cleaned_serial, {})
    if not game_meta:
        print(f"Error: Game metadata for serial {cleaned_serial} not found in master list.")
        sys.exit(1)

    new_mc2_name = f"{cleaned_serial}-01.mc2"
    new_mc2_path = os.path.join(TEMP_DIR, new_mc2_name)
    safe_rename(blank_mc2, new_mc2_path)

    blank_ps2 = extract_blank("BLANK.ps2")
    cmd_import_ps2 = [MYMCPLUSPLUS, blank_ps2, "import", temp_save_path]
    output_import_ps2 = run_command(cmd_import_ps2)
    print("PS2 Import Output:", output_import_ps2)

    save_folder = os.path.join(TEMP_DIR, "save")
    os.makedirs(save_folder, exist_ok=True)
    cmd_export_psu = [MYMCPLUSPLUS, blank_ps2, "export", "-p", "-f", "-l", "-d", save_folder, save_directory_full]
    output_export_psu = run_command(cmd_export_psu)
    print("PSU Export Output:", output_export_psu)

    exported_psu = None
    for f in os.listdir(save_folder):
        if f.lower().endswith(".psu"):
            exported_psu = os.path.join(save_folder, f)
            break
    if not exported_psu:
        print(f"Error: Exported .psu file not found in {save_folder}.")
        sys.exit(1)
    final_psu_path = exported_psu

    game_name = game_meta.get("name-en", game_meta.get("name", "Unknown Game"))
    region = game_meta.get("region", "UnknownRegion")
    new_ps2_name = f"{game_name} ({region}) [{cleaned_serial}].ps2"
    new_ps2_path = os.path.join(TEMP_DIR, new_ps2_name)
    safe_rename(blank_ps2, new_ps2_path)

    blank_bin = extract_blank("BLANK.bin")
    temp_psu_for_vmc = os.path.join(TEMP_DIR, "1.psu")
    safe_rename(final_psu_path, temp_psu_for_vmc)
    cmd_vmc = [ps2vmc_tool_path, blank_bin, "-pu", temp_psu_for_vmc]
    output_vmc = run_command(cmd_vmc)
    print("VMC Tool Output:", output_vmc)
    safe_rename(temp_psu_for_vmc, final_psu_path)

    bin_serial = cleaned_serial.replace("-", "_")
    bin_serial = bin_serial[:8] + "." + bin_serial[8:]
    new_bin_name = f"{bin_serial}_0.bin"
    new_bin_path = os.path.join(TEMP_DIR, new_bin_name)
    safe_rename(blank_bin, new_bin_path)
    
    final_base_folder = os.path.join(TEMP_DIR, game_name, username)
    serial_folder = os.path.join(final_base_folder, "01", cleaned_serial)
    os.makedirs(serial_folder, exist_ok=True)
    pcsx2_folder = os.path.join(final_base_folder, "01", "PCSX2")
    vmc_folder = os.path.join(final_base_folder, "01", "VMC")
    os.makedirs(pcsx2_folder, exist_ok=True)
    os.makedirs(vmc_folder, exist_ok=True)
    final_save_folder = os.path.join(final_base_folder, "01", "Save")
    shutil.move(save_folder, final_save_folder)

    # Perform the CRC32 check on all matching files in the game folder
    game_folder = os.path.join(TEMP_DIR, game_name)  # This is the correct game folder path
    files_to_check = find_files_recursive(game_folder, extensions=("bin", "ps2", "mc2"))

    # CRC32 values for the blank files from BLANKS.zip
    blank_crc32_values = {
        'bin': 0xB4DAF766,
        'mc2': 0x420463FF,
        'ps2': 0xA5D67EC6,
    }

    for file in files_to_check:
        file_extension = file.split('.')[-1]
        if file_extension in blank_crc32_values:
            file_crc32 = calculate_crc32(file)
            if file_crc32 == blank_crc32_values[file_extension]:
                print(f"Warning: CRC32 match detected for {file}. Aborting move to complete.")
                sys.exit(1)

    final_yaml_entry = {}
    if game_meta.get("name"):
        final_yaml_entry["name"] = game_meta["name"]
    if game_meta.get("name-sort"):
        final_yaml_entry["name-sort"] = game_meta["name-sort"]
    if game_meta.get("name-en"):
        final_yaml_entry["name-en"] = game_meta["name-en"]
    if region:
        final_yaml_entry["region"] = region
    if username:
        final_yaml_entry["user"] = username
    final_yaml_entry["date"] = date_str
    if description:
        final_yaml_entry["description"] = description
    if save_link:
        final_yaml_entry["save_link"] = save_link
    if game_meta.get("memcardFilters"):
        final_yaml_entry["memcardFilters"] = game_meta["memcardFilters"]

    txt_filename = f"{game_name} ({region}).txt"
    txt_filepath = os.path.join(TEMP_DIR, txt_filename)
    data_to_dump = {cleaned_serial: quote_values(final_yaml_entry)}
    with open(txt_filepath, 'w', encoding='utf-8', errors='replace') as f:
        yaml.dump(data_to_dump, f, allow_unicode=True, sort_keys=False)

    for folder in (serial_folder, pcsx2_folder, vmc_folder, final_save_folder):
        shutil.copy(txt_filepath, folder)
    if os.path.exists(os.path.join(TEMP_DIR, txt_filename)):
        os.remove(os.path.join(TEMP_DIR, txt_filename))

    shutil.move(new_mc2_path, serial_folder)
    shutil.move(new_ps2_path, pcsx2_folder)
    shutil.move(new_bin_path, vmc_folder)

    os.remove(temp_save_path)

    # Move the game folder (not the user folder) to the complete directory
    complete_folder = os.path.join(current_directory, 'complete')
    move_to_complete_folder(game_folder, complete_folder)  # Move the game folder to "complete"
    
    print("Memory card processing complete!")
    print(f"Final output is located in: {complete_folder}")
    
if __name__ == '__main__':
    main()
